
package be.arci.pub;

import be.arci.applet.*;
import java.awt.BorderLayout;
import java.awt.Dimension;
import javax.swing.JDialog;
import javax.swing.JEditorPane;
import javax.swing.JScrollPane;
import javax.swing.JViewport;
import javax.swing.JLabel;
import javax.swing.event.HyperlinkEvent;
import javax.swing.event.HyperlinkEvent.EventType;
import javax.swing.event.HyperlinkListener;
import javax.swing.text.html.HTMLFrameHyperlinkEvent;
import javax.swing.text.html.HTMLDocument;
import java.net.URL;
import java.io.IOException;

/**Implementation of Browser that uses a Swing JEditorPane to display 
 * HTML (and other) documents. 
 * <P>
 * We supply this Browser implementation as a free service to all Applicet users. 
 * You are allowed to modify this source code, provided you move the class out of 
 * the <tt>be.arci</tt> package hierarchy.
 * <P>
 * <STRONG>Currently the sTarget argument is ignored</STRONG>. 
 * <P>
 * A single SwingBrowser that is set as Browser displays all subsequent documents 
 * for <tt>Applicet.Context.showDocument()</tt>. If another Browser window is needed, 
 * a new SwingBrowser should be set as Applicet.Context's Browser. A SwingBrowser can
 * be shared by multiple Applicet applications that run together.
 * <P>
 * <STRONG>Note: </STRONG>Swing 1.1.1 on JDK118 doesn't seem to support
 * hyperlinks in frames.
 * @author Dirk Bosmans
 * @since 25/jan/2000
 */
public class SwingBrowser extends JEditorPane implements Browser, HyperlinkListener
{
 //---state and initialization-----------------------------------------
 /**Label to show the url of a hyperlink.
  * @serial can be null if dlg is null.  */
 private JLabel labHyperlink;
 /**Contains the SwingBrowser pane and the hyperlink label, for the default 
  * constructor; is null for the constructor with container argument. 
  * @serial dialog containing the SwingBrowser, or null */
 private JDialog dlg;
 /**Title of the Application we are currently displaying documents for. 
  * Set on each call to showDocument()  */
 private transient String sApplication;
 /**Sets up a new SwingBrowser in a JDialog.  */
 public SwingBrowser()
 {
  super();//just to make that clear
  setEditable(false);
  addHyperlinkListener(this);  
  JScrollPane sp = new JScrollPane();
  sp.getViewport().add(this);
  //don't parent dlg to applicet's parent frame; the browser dialog would obscure the applicet window
  //JDialog dlg = new JDialog(applicet.getFrame());
  dlg = new JDialog();
  Dimension dimScreen = dlg.getToolkit().getScreenSize();
  dlg.setSize(dimScreen.width / 2, dimScreen.height / 2);//arbitrary at 1/4 screen
  dlg.getContentPane().add(sp, BorderLayout.CENTER);
  labHyperlink = new JLabel(" ");//" " keeps label from popping away (do empty labels have preferredsize (0,0) ??)
  dlg.getContentPane().add(labHyperlink, BorderLayout.SOUTH);
 }
 /**Creates a SwingBrowser without adding it to a JDialog or any other 
  * containter; that is the responsibility of the caller. A JLabel argument
  * may be given to receive the URL of hyperlinks that the mouse cursor moves over.
  * @param labHyperlink label to receive the URL of hyperlinks that the mouse cursor 
  * moves over; this argument may be null.
  */
 public SwingBrowser(JLabel labHyperlink)
 {
  super();//just to make that clear
  setEditable(false);
  addHyperlinkListener(this);  
  this.labHyperlink = labHyperlink;
 }
 //---implements Browser-------------------------------------------
 /*javadoc copies docs from Browser*/
 public void showDocument(Applicet application, URL urlDocument, String sTarget)
 {
  sApplication = application.getFrame().getTitle();
  try {//catch all, because java.applet.AppletContext forbids to throw
   setPage(urlDocument);
  } catch (IOException e) { e.printStackTrace(); }
  if (dlg != null)//this is a SwingBrowser we hosted in a Container ourselves
   dlg.show();//dlg might have been hidden following a previous showDocument, so show it again
 }
 //---implements HyperlinkListener-------------------------------------
 /**Processes HyperlinkEvents (i.e. clicking on a hyperlink)
  * to display a new document; also displays the URL of the
  * hyperlink under the mouse in the labHyperlink label.
  */
 public void hyperlinkUpdate(HyperlinkEvent e) 
 {
  HyperlinkEvent.EventType heet = e.getEventType();
  if (heet == HyperlinkEvent.EventType.ACTIVATED) 
  {//currently we see no way to walk down a history list of HTMLFrameHyperlinkEvents, so we
   //just don't implement a 'forward-backward' browsing function at all
   if (e instanceof HTMLFrameHyperlinkEvent)  //HTMLFrameHyperlinkEvent should occur in HTMLDocument only
    ((HTMLDocument)getDocument()).processHTMLFrameHyperlinkEvent((HTMLFrameHyperlinkEvent)e);
   else 
   {
    try {
     setPage(e.getURL()); 
    } catch (IOException ioe) { ioe.printStackTrace(); }
   }
  }
  else if (heet == HyperlinkEvent.EventType.ENTERED)//note: FrameView should propagate HyperlinkEvent to here, but doesn't
  {
   if (labHyperlink != null)
    labHyperlink.setText(e.getURL().toExternalForm());
  }
  else if (heet == HyperlinkEvent.EventType.EXITED)
  {
   if (labHyperlink != null)
    labHyperlink.setText(" ");//" " keeps label from popping away (do empty labels have preferredsize (0,0) ??)
  }
 }
 //---workers-----------------------
 /**Sets the current URL being displayed, and adjusts the title of 
  * our containing dialog to the new document url.
  * @see "javax.swing.JEditorPane.setPage(URL page)"  */
 public void setPage(URL page) throws IOException 
 {
  super.setPage(page);
  //we would like to use the <TITLE> of the HTML document, but 
  //javax.swing.text.html.HTMLDocument is a work in progress, not 
  //safe to use such features already
  if (dlg != null)
  {
   try {//not essential, don't want to stumble over any stupid errors in this
    dlg.setTitle(sApplication + " -- " + page);
   } catch (Throwable t) { t.printStackTrace(); } //but still report it
  }
 }
}
